<?php
/**
 * Complete Assessment Percentage
 *
 * @package     AutomatorWP\Integrations\eLearnCommerce\Triggers\Complete_Assessment_Percentage
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_eLearnCommerce_Complete_Assessment_Percentage extends AutomatorWP_Integration_Trigger {

    public $integration = 'elearncommerce';
    public $trigger = 'elearncommerce_complete_assessment_score';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User completes a assessment with a grade percentage greater than, less than or equal to a specific percentage', 'automatorwp-pro' ),
            'select_option'     => __( 'User completes a assessment with a grade percentage <strong>greater than, less than or equal</strong> to a specific percentage', 'automatorwp-pro' ),
            /* translators: %1$s: Post title. %2$s: Condition. %3$s: Percent. %4$s: Number of times. */
            'edit_label'        => sprintf( __( 'User completes %1$s with a grade percentage %2$s %3$s %4$s time(s)', 'automatorwp-pro' ), '{post}', '{condition}', '{percentage}', '{times}' ),
            /* translators: %1$s: Post title. %2$s: Condition. %3$s: Percent. */
            'log_label'         => sprintf( __( 'User completes %1$s with a grade percentage %2$s %3$s', 'automatorwp-pro' ), '{post}', '{condition}', '{percentage}' ),
            'action'            => 'wpep_user_set_assessment_data_' . WPEP_USER_ASSESSMENT_SCORE,
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 5,
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name' => __( 'Assessment:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any assessment', 'automatorwp-pro' ),
                    'post_type' => WPEP_POST_TYPE_ASSESSMENT
                ) ),
                'condition' => automatorwp_utilities_condition_option(),
                'percentage' => array(
                    'from' => 'percentage',
                    'fields' => array(
                        'percentage' => array(
                            'name' => __( 'Percentage:', 'automatorwp-pro' ),
                            'desc' => __( 'The grade percentage required.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'attributes' => array(
                                'type' => 'number',
                                'min' => '0',
                            ),
                            'default' => 0
                        )
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param $grading
     * @param $assessment_id
     * @param $question_id
     * @param $question_answer_id
     * @param $user_id
     */
    public function listener( $grading, $assessment_id, $question_id, $question_answer_id, $user_id ) {

        // Bail if user is not set
        if( ! $user_id ) {
            return;
        }

        $grading = absint( $grading );

        // Bail if no score given
        if( $grading === 0 ) {
            return;
        }

        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'post_id'       => $assessment_id,
            'percentage'    => $grading,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['post_id'] ) && ! isset( $event['percentage'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( ! automatorwp_posts_matches( $event['post_id'], $trigger_options['post'] ) ) {
            return false;
        }

        $percentage = (float) $event['percentage'];
        $required_percentage =  (float) $trigger_options['percentage'];

        // Don't deserve if order total doesn't match with the trigger option
        if( ! automatorwp_number_condition_matches( $percentage, $required_percentage, $trigger_options['condition'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_eLearnCommerce_Complete_Assessment_Percentage();